<?php
/**
 * CAP-STONE HALL Results Portal - Main API
 * Handles all CRUD operations for the school results system
 */

require_once 'config.php';

$db = getDB();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

// Get action from query string
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        
        // ═══════════════════════════════════════════
        // AUTHENTICATION
        // ═══════════════════════════════════════════
        
        case 'login_teacher':
            $username = $input['username'] ?? '';
            $password = $input['password'] ?? '';
            
            $stmt = $db->prepare("SELECT * FROM teachers WHERE username = ?");
            $stmt->execute([$username]);
            $teacher = $stmt->fetch();
            
            if ($teacher && password_verify($password, $teacher['password'])) {
                unset($teacher['password']); // Don't send password hash
                $teacher['classes'] = json_decode($teacher['classes'], true);
                echo json_encode(['success' => true, 'teacher' => $teacher]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
            }
            break;
            
        case 'login_student':
            $student_id = $input['student_id'] ?? '';
            $pin = $input['pin'] ?? '';
            
            $stmt = $db->prepare("SELECT * FROM students WHERE id = ? AND pin = ?");
            $stmt->execute([$student_id, $pin]);
            $student = $stmt->fetch();
            
            if ($student) {
                echo json_encode(['success' => true, 'student' => $student]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Invalid PIN']);
            }
            break;
        
        // ═══════════════════════════════════════════
        // STUDENTS
        // ═══════════════════════════════════════════
        
        case 'get_students':
            $class_id = $_GET['class_id'] ?? '';
            
            if ($class_id) {
                $stmt = $db->prepare("SELECT * FROM students WHERE class_id = ? ORDER BY first_name, other_names");
                $stmt->execute([$class_id]);
            } else {
                $stmt = $db->query("SELECT * FROM students ORDER BY class_id, first_name, other_names");
            }
            
            echo json_encode(['students' => $stmt->fetchAll()]);
            break;
            
        case 'add_student':
            $id = generateId('s');
            $stmt = $db->prepare("INSERT INTO students (id, first_name, other_names, class_id, sex, pin) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $id,
                $input['first_name'],
                $input['other_names'] ?? '',
                $input['class_id'],
                $input['sex'],
                $input['pin']
            ]);
            echo json_encode(['success' => true, 'id' => $id]);
            break;
            
        case 'update_student':
            $stmt = $db->prepare("UPDATE students SET first_name = ?, other_names = ?, class_id = ?, sex = ?, pin = ? WHERE id = ?");
            $stmt->execute([
                $input['first_name'],
                $input['other_names'] ?? '',
                $input['class_id'],
                $input['sex'],
                $input['pin'],
                $input['id']
            ]);
            echo json_encode(['success' => true]);
            break;
            
        case 'delete_student':
            $stmt = $db->prepare("DELETE FROM students WHERE id = ?");
            $stmt->execute([$input['id']]);
            echo json_encode(['success' => true]);
            break;
        
        // ═══════════════════════════════════════════
        // TEACHERS
        // ═══════════════════════════════════════════
        
        case 'get_teachers':
            $stmt = $db->query("SELECT id, name, username, is_admin, classes FROM teachers ORDER BY name");
            $teachers = $stmt->fetchAll();
            foreach ($teachers as &$teacher) {
                $teacher['classes'] = json_decode($teacher['classes'], true);
            }
            echo json_encode(['teachers' => $teachers]);
            break;
            
        case 'add_teacher':
            $id = generateId('t');
            $stmt = $db->prepare("INSERT INTO teachers (id, name, username, password, is_admin, classes) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $id,
                $input['name'],
                $input['username'],
                password_hash($input['password'], PASSWORD_BCRYPT),
                $input['is_admin'] ? 1 : 0,
                json_encode($input['classes'] ?? [])
            ]);
            echo json_encode(['success' => true, 'id' => $id]);
            break;
            
        case 'update_teacher':
            if (isset($input['password']) && $input['password']) {
                $stmt = $db->prepare("UPDATE teachers SET name = ?, username = ?, password = ?, is_admin = ?, classes = ? WHERE id = ?");
                $stmt->execute([
                    $input['name'],
                    $input['username'],
                    password_hash($input['password'], PASSWORD_BCRYPT),
                    $input['is_admin'] ? 1 : 0,
                    json_encode($input['classes'] ?? []),
                    $input['id']
                ]);
            } else {
                $stmt = $db->prepare("UPDATE teachers SET name = ?, username = ?, is_admin = ?, classes = ? WHERE id = ?");
                $stmt->execute([
                    $input['name'],
                    $input['username'],
                    $input['is_admin'] ? 1 : 0,
                    json_encode($input['classes'] ?? []),
                    $input['id']
                ]);
            }
            echo json_encode(['success' => true]);
            break;
            
        case 'delete_teacher':
            $stmt = $db->prepare("DELETE FROM teachers WHERE id = ?");
            $stmt->execute([$input['id']]);
            echo json_encode(['success' => true]);
            break;
        
        // ═══════════════════════════════════════════
        // SUBJECTS
        // ═══════════════════════════════════════════
        
        case 'get_subjects':
            $class_id = $_GET['class_id'] ?? '';
            $stmt = $db->prepare("SELECT subject_name FROM class_subjects WHERE class_id = ? ORDER BY display_order, subject_name");
            $stmt->execute([$class_id]);
            echo json_encode(['subjects' => $stmt->fetchAll(PDO::FETCH_COLUMN)]);
            break;
            
        case 'update_subjects':
            // Delete existing subjects for this class
            $stmt = $db->prepare("DELETE FROM class_subjects WHERE class_id = ?");
            $stmt->execute([$input['class_id']]);
            
            // Insert new subjects
            $stmt = $db->prepare("INSERT INTO class_subjects (class_id, subject_name, display_order) VALUES (?, ?, ?)");
            foreach ($input['subjects'] as $index => $subject) {
                $stmt->execute([$input['class_id'], $subject, $index]);
            }
            echo json_encode(['success' => true]);
            break;
        
        // ═══════════════════════════════════════════
        // SCORES (Complete student record)
        // ═══════════════════════════════════════════
        
        case 'get_student_scores':
            $student_id = $_GET['student_id'] ?? '';
            
            // Get scores
            $stmt = $db->prepare("SELECT * FROM student_scores WHERE student_id = ?");
            $stmt->execute([$student_id]);
            $scores = $stmt->fetchAll();
            
            // Get behaviours
            $stmt = $db->prepare("SELECT * FROM student_behaviours WHERE student_id = ?");
            $stmt->execute([$student_id]);
            $behaviours = $stmt->fetchAll();
            
            // Get skills
            $stmt = $db->prepare("SELECT * FROM student_skills WHERE student_id = ?");
            $stmt->execute([$student_id]);
            $skills = $stmt->fetchAll();
            
            // Get attendance & comments
            $stmt = $db->prepare("SELECT * FROM student_records WHERE student_id = ?");
            $stmt->execute([$student_id]);
            $records = $stmt->fetch();
            
            echo json_encode([
                'scores' => $scores,
                'behaviours' => $behaviours,
                'skills' => $skills,
                'records' => $records ?: []
            ]);
            break;
            
        case 'save_student_scores':
            $student_id = $input['student_id'];
            
            // Save scores (subject by subject)
            if (isset($input['subjects'])) {
                $stmt = $db->prepare("INSERT INTO student_scores (student_id, subject_name, ra_test, mid_term, random_test, practicals, exam) 
                    VALUES (?, ?, ?, ?, ?, ?, ?) 
                    ON DUPLICATE KEY UPDATE ra_test=VALUES(ra_test), mid_term=VALUES(mid_term), random_test=VALUES(random_test), practicals=VALUES(practicals), exam=VALUES(exam)");
                
                foreach ($input['subjects'] as $subject => $scores) {
                    $stmt->execute([
                        $student_id,
                        $subject,
                        $scores['ra'] ?? 0,
                        $scores['mid'] ?? 0,
                        $scores['rand'] ?? 0,
                        $scores['pract'] ?? 0,
                        $scores['exam'] ?? 0
                    ]);
                }
            }
            
            // Save behaviours
            if (isset($input['behaviours'])) {
                $stmt = $db->prepare("INSERT INTO student_behaviours (student_id, behaviour_name, rating) 
                    VALUES (?, ?, ?) 
                    ON DUPLICATE KEY UPDATE rating=VALUES(rating)");
                
                foreach ($input['behaviours'] as $behaviour => $rating) {
                    $stmt->execute([$student_id, $behaviour, $rating]);
                }
            }
            
            // Save skills
            if (isset($input['skills'])) {
                $stmt = $db->prepare("INSERT INTO student_skills (student_id, skill_name, rating) 
                    VALUES (?, ?, ?) 
                    ON DUPLICATE KEY UPDATE rating=VALUES(rating)");
                
                foreach ($input['skills'] as $skill => $rating) {
                    $stmt->execute([$student_id, $skill, $rating]);
                }
            }
            
            // Save attendance & comments
            if (isset($input['attendance']) || isset($input['resumption']) || isset($input['cmt1']) || isset($input['cmt2'])) {
                $stmt = $db->prepare("INSERT INTO student_records (student_id, days_opened, days_present, resumption_date, instructor_comment, principal_comment) 
                    VALUES (?, ?, ?, ?, ?, ?) 
                    ON DUPLICATE KEY UPDATE days_opened=VALUES(days_opened), days_present=VALUES(days_present), resumption_date=VALUES(resumption_date), instructor_comment=VALUES(instructor_comment), principal_comment=VALUES(principal_comment)");
                
                $stmt->execute([
                    $student_id,
                    $input['attendance']['opened'] ?? 0,
                    $input['attendance']['present'] ?? 0,
                    $input['resumption'] ?? '',
                    $input['cmt1'] ?? '',
                    $input['cmt2'] ?? ''
                ]);
            }
            
            echo json_encode(['success' => true]);
            break;
        
        // ═══════════════════════════════════════════
        // SCHOOL SETTINGS
        // ═══════════════════════════════════════════
        
        case 'get_settings':
            $stmt = $db->query("SELECT setting_key, setting_value FROM school_settings");
            $settings = [];
            foreach ($stmt->fetchAll() as $row) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
            echo json_encode(['settings' => $settings]);
            break;
            
        case 'update_settings':
            $stmt = $db->prepare("INSERT INTO school_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value=VALUES(setting_value)");
            
            foreach ($input as $key => $value) {
                $stmt->execute([$key, $value]);
            }
            echo json_encode(['success' => true]);
            break;
        
        // ═══════════════════════════════════════════
        // DASHBOARD STATS
        // ═══════════════════════════════════════════
        
        case 'get_dashboard_stats':
            // Total students
            $total_students = $db->query("SELECT COUNT(*) FROM students")->fetchColumn();
            
            // Students with scores entered
            $students_with_scores = $db->query("SELECT COUNT(DISTINCT student_id) FROM student_scores")->fetchColumn();
            
            // Overall average
            $overall_avg = $db->query("SELECT AVG(total) FROM student_scores WHERE total > 0")->fetchColumn();
            
            // Class breakdown
            $stmt = $db->query("
                SELECT s.class_id, COUNT(DISTINCT s.id) as total, 
                       COUNT(DISTINCT sc.student_id) as with_scores,
                       AVG(sc.total) as class_avg
                FROM students s
                LEFT JOIN student_scores sc ON s.id = sc.student_id
                GROUP BY s.class_id
            ");
            
            echo json_encode([
                'total_students' => $total_students,
                'students_with_scores' => $students_with_scores,
                'overall_avg' => round($overall_avg ?? 0, 1),
                'classes' => $stmt->fetchAll()
            ]);
            break;
        
        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action']);
    }
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}
?>
